function [rxTxTraj] = waymarkTrajGenSwellex(time)
% A function to generate the depth of the receiver and transmitter and the range between them over time.  This data is from the SWellEx-96 experiment with an estimate of the trajectory created from the following papers;

% Referenes
% =========
% [1] T. Chen, C. Liu, and Y. V. Zakharov, "Matched-phase coherent broadband matched-field processor using phase descent search," in Proc. 10th ECUA, 2010, pp. 590?595.
% [2] C. Liu, T. Chen, and Y. V. Zakharov, "Source localization using sparsitybased iterative adaptive beamforming," in Proc. 10th ECUA, 2010, pp. 604?610.

% Where:
%	rxTxTraj.rxDepth, receiver depths at each waymark (m)
%	rxTxTraj.txDepth, transmitter depths at each waymark (m)
%	rxTxTraj.rxTxRange, range between the receiver and transmitter at each waymark (m)

clockTimePts		= 9*60;		% Clock time through the experiment (s)
deltaTime			= 100;		% Size of the time window for the calculations (s)
interpIndex			= 1:1:deltaTime+1;
% Gate the data removing point that exceed the estimated speed of the receiver
RANGE_DIFF_LIMIT	= 0.02;		% Range difference limit (km/s)

% Extract the position information from the localisation results (obtained from the work reported [1] and [2])
% Load the generated data, the SWellEx test is 75 minutes long and a localisation every second
swellex = load('range_depth_start_end.mat');

rangeInterp = zeros(size(clockTimePts));
depthInterp = zeros(size(clockTimePts));

for clockTimeIndex = 1:length(clockTimePts)

	% Specify the start and end time so that the time of interest is at the centre
	tStart	= clockTimePts(clockTimeIndex) - floor(deltaTime/2);
	tEnd		= tStart + deltaTime;
	deltaTimeIndex = (tStart:1:tEnd);

	rangeDeltaTime = swellex.range(deltaTimeIndex);
	depthDeltaTime = swellex.depth(deltaTimeIndex);

	rangeDiff			= rangeDeltaTime(2:end) - rangeDeltaTime(1:end-1);	% Find the difference between results
	rangeLimitIndex	= find(abs(rangeDiff) >= RANGE_DIFF_LIMIT);
	rangeLimitIndex	= [rangeLimitIndex, rangeLimitIndex+1, rangeLimitIndex-1];
	rangeLimitIndex	= rangeLimitIndex((rangeLimitIndex >= 1));
	
	% Limit the change in the range
	rangeLimited = rangeDeltaTime;
	rangeLimited(rangeLimitIndex) = [];			% Remove the elements greater than the limit
	
	% Remove the corresponding depth measurments
	depthLimited = depthDeltaTime;
	depthLimited(rangeLimitIndex) = [];			% Remove the elements greater than the limit
	
	clockTime = (tStart:1:tEnd);
	clockTimeLimited = (tStart:1:tEnd);
	clockTimeLimited(rangeLimitIndex) = [];	% Remove the corresponding time elements


	polyCoEffRange = round(polyfit(clockTimeLimited, rangeLimited*1000, 1)*10)/10;	% Linear fit the range using localization results, convert to meters (*1000) for the interpolation.  Round the result to one decimal place
	rangeInterp		= round(polyval(polyCoEffRange, clockTime));								% Reconstruct the signal

	depthInterp		= round(mean(depthLimited));

end

% % Plots
% % =====
% rangeFig = figure('name', 'Range Localisation - Range Interp');
% hold on;
% subplot(3,1,1);
% plot(swellex.range, 'b');
% subplot(3,1,2);
% plot(clockTimeLimited, rangeLimited, 'r');
% subplot(3,1,3);
% plot(clockTime, rangeInterp, 'b');
% 
% depthFig = figure('name', 'Depth Localisation - Depth Interp');
% hold on;
% subplot(3,1,1);
% plot(-swellex.depth, 'b');
% subplot(3,1,2);
% plot(clockTimeLimited, -depthLimited, 'g');
% subplot(3,1,3);
% plot(clockTime, -depthInterp, 'b');


% Waymark trajectory generation as "waymarkTrajGenBellhopFunc.m" but takes the parameters from the calculation above.

% Generate the receiver depth
% ===========================
% Constant receiver depth
RX_A = 155.99;								% Depth in meters.  This is element 31 of the VLA which is the closest to halfway down the array (153.1875 m)
rxTxTraj.rxDepth = RX_A;


% Generate the transmitter depth
% ==============================
% Constant transmitter depth
rxTxTraj.txDepth = depthInterp;


% Generate the receiver - transmitter range
% =========================================
% Generate a simple straight line rxTxRange = Ax + B
rxTxTraj.rxTxRange = (polyCoEffRange(1)*time) + polyCoEffRange(2);	% generate waymark depths
